﻿using System.Drawing;

namespace GuitarTrainer.GUI
{
    /// <summary>
    /// Contains information regarding various display options used by a particular
    /// song and the entire program in general
    /// </summary>
    internal class DisplayOptions
    {
        // When there is a unique color
        // Are the color of the rhythm/notes related to the duration
        public const int RELATED_TO_DURATION = 1;
        // Are the color of the rhythm/notes related to the dynamic
        public const int RELATED_TO_DYNAMIC = 2;

        // The spacing between the horizontal lines in pixels
        // How much space below is preserved
        private int bottomOffset;

        // Are the lines extended at the top?

        // Is the music cursor displayed?
        private bool displayMusicCursor;
        // Display a question mark when an effect is present in the song but not yet drawn
        private bool displayUnsupportedEffects;
        private bool extendBottom;
        private bool extendTop;
        private int fretColoringType;
        private ColorScheme fretColors;
        private int lineSpacing;
        // Is proportional spacing linked to the piece?
        private bool minPieceSpacing;
        // Is the preffered view multi-track?
        private bool multiTrackView;

        // If true, the music cursors are all in the same color.
        // If false, each music cursor is painted the same color as its track's color

        private Color musicCursorColor;
        private int noteSpacing;
        private int rhythmColoringType;
        private ColorScheme rhythmColors;
        private bool singleMusicCursorColor;
        private int topOffset;

        public DisplayOptions()
        {
            // 10 is the default value for GuitarPro files
            lineSpacing = 10;
            noteSpacing = 3;

            topOffset = 0;
            bottomOffset = 0;
            extendBottom = false;
            extendTop = false;

            rhythmColors = new ColorScheme();
            fretColors = new ColorScheme();

            rhythmColoringType = RELATED_TO_DURATION;
            fretColoringType = RELATED_TO_DURATION;
            displayMusicCursor = true;
            displayUnsupportedEffects = true;

            minPieceSpacing = false;
            multiTrackView = true;

            singleMusicCursorColor = false;
            musicCursorColor = Color.DarkGray;
        }

        public DisplayOptions clone()
        {
            var toReturn = new DisplayOptions
                               {
                                   LineSpacing = LineSpacing,
                                   NoteSpacing = NoteSpacing,
                                   TopOffset = TopOffset,
                                   BottomOffset = BottomOffset,
                                   ExtendBottom = ExtendBottom,
                                   ExtendTop = ExtendTop,
                                   RhythmColors = RhythmColors.clone(),
                                   FretColors = FretColors.clone(),
                                   RhythmColoringType = RhythmColoringType,
                                   FretColoringType = FretColoringType,
                                   DisplayMusicCursor = DisplayMusicCursor,
                                   DisplayUnsupportedEffects = DisplayUnsupportedEffects,
                                   MinPieceSpacing = MinPieceSpacing,
                                   MultiTrackView = MultiTrackView,
                                   SingleMusicCursorColor = SingleMusicCursorColor,
                                   MusicCursorColor = MusicCursorColor
                               };

            return toReturn;
        }

        /// <summary>
        /// Switch back and forth between the two types of spacing
        /// </summary>
        public void toggleMinPieceSpacing()
        {
            minPieceSpacing = !minPieceSpacing;
        }

        /// <summary>
        /// Switch between displaying and not displaying unsupported effects
        /// </summary>
        public void toggleDisplayOfUnsupportedEffects()
        {
            displayMusicCursor = !displayMusicCursor;
        }

        /// <summary>
        /// Change whether or not the music cursor should be one color only or not
        /// </summary>
        public void toggleSingleMusicCursorColor()
        {
            singleMusicCursorColor = !singleMusicCursorColor;
        }

        /// <summary>
        /// Switch between enabling and disable the music cursor
        /// </summary>
        public void toggleViewMusicCursor()
        {
            displayMusicCursor = !displayMusicCursor;
        }

        /// <summary>
        /// Switch between whether or not multiple tracks should be displayed
        /// </summary>
        public void toggleMultiTrackView()
        {
            multiTrackView = !multiTrackView;
        }

        #region Getters and setters

        public int LineSpacing
        {
            get { return lineSpacing; }
            private set { lineSpacing = value; }
        }

        public int NoteSpacing
        {
            get { return noteSpacing; }
            set { noteSpacing = value; }
        }

        public int TopOffset
        {
            get { return topOffset; }
            set { topOffset = value; }
        }

        public int BottomOffset
        {
            get { return bottomOffset; }
            set { bottomOffset = value; }
        }

        public bool ExtendBottom
        {
            get { return extendBottom; }
            set { extendBottom = value; }
        }

        public bool ExtendTop
        {
            get { return extendTop; }
            set { extendTop = value; }
        }

        public ColorScheme RhythmColors
        {
            get { return rhythmColors; }
            private set { rhythmColors = value; }
        }

        public ColorScheme FretColors
        {
            get { return fretColors; }
            private set { fretColors = value; }
        }

        private int RhythmColoringType
        {
            get { return rhythmColoringType; }
            set { rhythmColoringType = value; }
        }

        public int FretColoringType
        {
            get { return fretColoringType; }
            private set { fretColoringType = value; }
        }

        public bool DisplayMusicCursor
        {
            get { return displayMusicCursor; }
            private set { displayMusicCursor = value; }
        }

        public bool DisplayUnsupportedEffects
        {
            get { return displayUnsupportedEffects; }
            private set { displayUnsupportedEffects = value; }
        }

        public bool MinPieceSpacing
        {
            get { return minPieceSpacing; }
            set { minPieceSpacing = value; }
        }

        public bool MultiTrackView
        {
            get { return multiTrackView; }
            set { multiTrackView = value; }
        }

        public bool SingleMusicCursorColor
        {
            get { return singleMusicCursorColor; }
            private set { singleMusicCursorColor = value; }
        }

        public Color MusicCursorColor
        {
            get { return musicCursorColor; }
            set { musicCursorColor = value; }
        }

        #endregion
    }
}