﻿using System;
using System.Drawing;
using System.Windows.Forms;

namespace GuitarTrainer.GUI
{
    /// <summary>
    /// Class for representing the music cursor bar (to let the user of the application
    /// know where in the song they are).
    /// </summary>
    class MusicCursor : Panel
    {
        private MusicCursorPosition musicCursorPosition;

        private int leftOffset = 8;
        private int rightOffset = 10;

        private bool firstTime;

        /// <summary>
        /// Creates a cursor located at measure 1, beat 1 and links it to a bar
        /// </summary>
        /// <param name="bp">The bar panel to associate this cursor with</param>
        public MusicCursor(BarPanel bp)
        {
            if (bp != null)
            {
                musicCursorPosition = new MusicCursorPosition {Width = leftOffset + rightOffset};

                firstTime = true;
            }
            else
            {
                MessageBox.Show("The BarPanel passed is null");
                Environment.Exit(0);
            }
        }

        public override Size MinimumSize
        {
            get
            {
                int h = Height - 2;

                return new Size(musicCursorPosition.Width, h);
            }
            set
            {
                base.MinimumSize = value;
            }
        }

        /// <summary>
        /// Controls how this cursor will be painted when needed
        /// </summary>
        /// <param name="e">The event parameters initialized when this event occurs</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            Point bottomLineA;
            Point bottomLineB;

            if (firstTime)
            {
                bottomLineA = new Point(e.ClipRectangle.Left - 100, e.ClipRectangle.Bottom - 10);
                bottomLineB = new Point(e.ClipRectangle.Right - 100, e.ClipRectangle.Bottom - 10);
                firstTime = false;
            }
            else
            {
                bottomLineA = new Point(e.ClipRectangle.Left/2 + 120, e.ClipRectangle.Bottom);
                bottomLineB = new Point(e.ClipRectangle.Right/2 + 200, e.ClipRectangle.Bottom);
            }
            
            Pen linePen = new Pen(new SolidBrush(Color.Red), 5.0f);
            e.Graphics.DrawLine(linePen, bottomLineA.X, bottomLineA.Y, bottomLineB.X, bottomLineB.Y);
        }

        /// <summary>
        /// Needs to be called each time the field point is updated
        /// </summary>
        private void pointUpdated()
        {
            int h = Height;
            int w = musicCursorPosition.Width;
            Point p = musicCursorPosition.Point;

            SetBounds(p.X - leftOffset, p.Y, w, h);
        }


        #region Getters and setters
        public MusicCursorPosition MusicCursorPosition
        {
            set
            {
                musicCursorPosition = value;
                pointUpdated();
            }
        }
        public int LeftOffset
        {
            get { return leftOffset; }
        }
        #endregion
    }
}
