﻿using System;
using System.Drawing;
using System.Windows.Forms;
using GuitarTrainer.GuitarProComponents;

namespace GuitarTrainer.GUI
{
    /// <summary>
    /// Creates and drawas different types of slides between notes
    /// </summary>
    class SlideLabel : Label
    {
        private readonly DisplayOptions dispOptions;
        private int typeOfSlide;
        private const int SLIDE_SHIFT = 1;
        private const int SLIDE_LEGATO = 2;
        private const int SLIDE_IN_FROM_BELOW = 3;
        private const int SLIDE_IN_FROM_ABOVE = 4;
        private const int SLIDE_OUT_UPWARDS = 5;
        private const int SLIDE_OUT_DOWNWARDS = 6;

        // To set the tool tips, these were taken from GPSlide
        public static string[] slideNames = { "Shift slide", "Legato slide",
            "Slide in from below", "Slide in from above", "Slide out upwards",
            "Slide out downwards" };

        // The color slides will appear in
        private readonly Color color;

        private int width;
        private const int MIN_WIDTH = 5;

        //FlowLayoutPanel panel;

        private Label label;

        /// <summary>
        /// Constructor for use with casting
        /// </summary>
        public SlideLabel()
        {
            Label = new Label();
            typeOfSlide = SLIDE_SHIFT;
            dispOptions = null;
            color = Color.Pink;
        }

        /// <summary>
        /// Creates a slide label that can be linked with a GPSlide object
        /// </summary>
        /// <param name="currDispOps">The display options to be used</param>
        /// <param name="desiredWidth">The width desired</param>
        public SlideLabel(DisplayOptions currDispOps, int desiredWidth)
        {
            if ((currDispOps != null) && (desiredWidth >= MIN_WIDTH))
            {
                // Set as the default slide type, for now
                typeOfSlide = SLIDE_SHIFT;
                width = desiredWidth;
                dispOptions = currDispOps;
                label = new Label();

                color = Color.Blue;
            }
            else
            {
                if (currDispOps == null)
                {
                    MessageBox.Show("Display options were null (SlideLabel())");
                    Environment.Exit(0);
                }
                else
                {
                    MessageBox.Show("Width was too small (SlideLabel())");
                    Environment.Exit(0);
                }
            }
        }

        /// <summary>
        /// Finds the correct location for the slide
        /// </summary>
        /// <param name="anchor">The point where the slide to be drawn begins from</param>
        /// <returns>A point where the bounds of the slide can be set</returns>
        private Point getPoint(Point anchor)
        {
            int deltaX, deltaY;

            int w = width - 1;
            int h = dispOptions.LineSpacing - 1;

            if ((typeOfSlide == SLIDE_IN_FROM_ABOVE) || (typeOfSlide == SLIDE_IN_FROM_BELOW))
            {
                deltaX = -w - 5;
                if (typeOfSlide == SLIDE_IN_FROM_BELOW)
                    deltaY = 0;
                else
                    deltaY = -h;
            }
            else
            {
                deltaX = 5;
                if (typeOfSlide == SLIDE_OUT_DOWNWARDS)
                    deltaY = 0;
                else
                    deltaY = -h;
            }

            Point p = new Point(anchor.X + deltaX, anchor.Y + deltaY);

            return p;
        }

        /// <summary>
        /// Draw a slide in the label.
        /// 
        /// Currently this is not drawn on the fly but most of the pieces are here for it. 
        /// It only needs a few more tweaks to become usable
        /// </summary>
        /// <param name="g">The graphics object to draw this slide label on</param>
        private void drawSlide(Graphics g)
        {
            Point p1 = new Point();
            Point p2 = new Point();
            int h = dispOptions.LineSpacing;
            int w = width - 1;

            p1.X = 0;
            p2.X = w;

            if ((typeOfSlide == SLIDE_IN_FROM_ABOVE) || (typeOfSlide == SLIDE_OUT_DOWNWARDS))
            {
                p1.Y = 0;
                p2.Y = h;
            }
            else
            {
                p1.Y = h;

                if (typeOfSlide == SLIDE_LEGATO)
                    p2.Y = h / 2;
                else
                    p2.Y = 0;
            }

            g.DrawLine(new Pen(new SolidBrush(color)), p1, p2);

            if (typeOfSlide == SLIDE_LEGATO)
                g.DrawArc(new Pen(new SolidBrush(color)), new Rectangle(0, 0, w, h), 0, 180);
        }

        /// <summary>
        ///  What to do when a slide label needs to be drawn
        /// </summary>
        /// <param name="e">The event information when this event occurs</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            Graphics g = e.Graphics;

            drawSlide(e.Graphics);

            Point p1 = new Point();
            Point p2 = new Point();
            int h = dispOptions.LineSpacing;
            int w = width - 1;

            p1.X = 0;
            p2.X = w;

            if ((typeOfSlide == SLIDE_IN_FROM_ABOVE) || (typeOfSlide == SLIDE_OUT_DOWNWARDS))
            {
                p1.Y = 0;
                p2.Y = h;
            }
            else
            {
                p1.Y = h;

                if (typeOfSlide == SLIDE_LEGATO)
                    p2.Y = h / 2;
                else
                    p2.Y = 0;
            }

            g.DrawLine(new Pen(new SolidBrush(color)), p1, p2);

            if (typeOfSlide == SLIDE_LEGATO)
                g.DrawArc(new Pen(new SolidBrush(color)), new Rectangle(0, 0, w, h), 0, 180);
        }

        #region Getters and setters

        private Point getPoint(int x, int y)
        {
            return getPoint(new Point(x, y));
        }
        public void setAnchor(Point anchor)
        {
            Point p = getPoint(anchor);
            Location = p;
        }
        public void setAnchor(int x, int y)
        {
            Point p = getPoint(x,y);
            label.Location = p;
        }

        private void setTypeOfSlide(int value)
        {
            if ((value >= SLIDE_SHIFT) && (value <= SLIDE_OUT_DOWNWARDS))
            {
                typeOfSlide = value;
                if (typeOfSlide == SLIDE_LEGATO)
                    Size = new Size(width, 2 * dispOptions.LineSpacing);
            }
            else
            {
                MessageBox.Show("Invalid range for setting slide (SlideLabel.TypeOfSlide set)");
                Environment.Exit(0);
            }
        }
        public void setTypeOfSlide(GPSlide value)
        {
            int type = -99;
            int gpsType = value.getIndex();

            switch (gpsType)
            {
                case 0:
                    type = SLIDE_IN_FROM_ABOVE;
                    break;
                case 1:
                    type = SLIDE_IN_FROM_BELOW;
                    break;
                case 2:
                    type = SLIDE_SHIFT;
                    break;
                case 3:
                    type = SLIDE_LEGATO;
                    break;
                case 4:
                    type = SLIDE_OUT_DOWNWARDS;
                    break;
                case 5:
                    type = SLIDE_OUT_UPWARDS;
                    break;
            }
            setTypeOfSlide(type);
        }

        private Label Label
        {
            set { label = value; }
        }

        #endregion
    }
}
