﻿using System;
using System.Drawing;
using System.Windows.Forms;
using GuitarTrainer.GuitarProComponents;
using GuitarTrainer.GP4File;
using GuitarTrainer.Sound;
using GuitarTrainer.Interfaces;

namespace GuitarTrainer.GUI
{
    sealed class SongInternalFrame : Panel
    {
        private readonly Form1 parent;
        private GPSong gpSong;
        private readonly SongPanel songPanel;
        private readonly DisplayOptions displayOptions;

        private MasterPlayer player;

        /// <summary>Used to evaluate performance when a process starts</summary>
        private readonly DateTime start;
        /// <summary>Used to evaluate performance when a process ends</summary>
        private DateTime finish;

        /// <summary>
        /// Constructor linked to the main window
        /// </summary>
        /// <param name="f1">The window that contains this one</param>
        /// <param name="dos">The display options used by this class</param>
        /// <param name="start">The start time of when this song was loaded</param>
        public SongInternalFrame(Form1 f1, DisplayOptions dos, DateTime start)
        {
            parent = f1;
            this.start = start;
            displayOptions = dos.clone();

            AutoScroll = true;
            Location = new Point(0, 20);
            Width = 1005;

            songPanel = new SongPanel(f1);
            setDisplayOptions();

            Controls.Add(songPanel);
            songPanel.Visible = true;
        }

        /// <summary>
        /// Sets the display location of the song internal frame
        /// </summary>
        /// <param name="rect">The rectangle that the SIF will take</param>
        public void setDisplayLocation(Rectangle rect)
        {
            AutoScrollPosition = new Point(rect.X, rect.Y);
        }

        /// <summary>
        /// Initiates the song by reading it in and sets up the events which, for the most
        /// part, are not implenting or working
        /// </summary>
        private void initSongAndPlayer()
        {
            if (Song == null)
            {
                try
                {
                    Song = GPAdaptor.makeSong(gpSong);

                    player = new MasterPlayer();
                }
                catch (Exception e)
                {
                    MessageBox.Show("Error with initSongAndPlayer() (SongInternalFrame)\n" + e);
                    Environment.Exit(0);
                }
            }
            player.TimerFrequency = 4;
            player.EnableNoteEvents = true;

            bool multiTrack = songPanel.MultiTrackView;
            for(int i = 0; i < songPanel.NumberOfTracks; i++)
            {
                bool scrollTrack = ((!multiTrack) || (i == 0));
                new MusicCursorScroller(songPanel.getTrackPanel(i), scrollTrack);
            }

            finish = DateTime.Now;

            if(parent.evaluate)
            {
                finish = DateTime.Now;

                TimeSpan span = finish.Subtract(start);
                MessageBox.Show(span.Seconds + " seconds");
            }

            setStatus(Form1.NOT_PLAYING);
        }

        /// <summary>
        /// Whether or not the object is equal to this SIF
        /// </summary>
        /// <param name="obj">The object to test</param>
        /// <returns>True if the object is equal, false otherwise</returns>
        public override bool Equals(object obj)
        {
            bool equal = false;
            SongInternalFrame temp;

            if (obj != null)
            {
                if (obj.GetType().IsInstanceOfType(this))
                {
                    temp = (SongInternalFrame)obj;
                    if (temp.Name == null)
                        equal = Name == null;
                    else
                        equal = temp.Name.Equals(Name);
                }
            }

            return equal;
        }

        /// <summary>
        /// Reads a GuitarPro file in then begins the drawing process
        /// </summary>
        /// <param name="inStr">The input stream to read the GuitarPro file from</param>
        /// <returns>Returns true if everything was successful, false otherwise</returns>
        public bool readAndDisplay(GP4InputStream inStr)
        {
            bool success = songPanel.readPieceFromFile(inStr);

            if (success)
            {
                gpSong = songPanel.Piece;
                songPanel.drawPiece();
                songPanel.Invalidate();
            }
            Height = songPanel.Height;
            return success;
        }

        private void setDisplayOptions()
        {
            songPanel.DisplayOptions = displayOptions;
        }

        public void setStatus(short stat)
        {
            switch (stat)
            {
                case Form1.PLAYING:
                    initSongAndPlayer();
                    break;
                case Form1.NOT_PLAYING:
                    break;
            }
        }

        public SongPanel SongPanel
        {
            get { return songPanel; }
        }

        public ISong Song { get; private set; }
    }
}
