﻿using System;
using System.Collections.Generic;
using System.Windows.Forms;
using System.Drawing;
using System.IO;
using GuitarTrainer.GuitarProComponents;
using GuitarTrainer.GP4File;

namespace GuitarTrainer.GUI
{
    /// <summary>
    /// Draws a panel on the form
    /// </summary>
    sealed class SongPanel : FlowLayoutPanel
    {
        private List<GPMeasure> measures;
        private List<GPTrack> tracks;

        // Iterate over TrackPanel collection
        private int count;

        // If tracks have been added
        private bool hasTracks;

        // If the view has changed
        private readonly bool viewChanged;

        // If the spacing has changed
        private bool spacingChanged;

        private DisplayOptions displayOptions;

        // How many lines are to be added to the top
        private const int LINES_ON_TOP = 2;
        // How many lines are to be added to the bottom
        private const int LINES_ON_BOTTOM = 3;

        private readonly Form1 superParent;

        /// <summary>
        /// Default constructor.
        /// </summary>
        public SongPanel(Form1 f1)
        {
            displayOptions = new DisplayOptions();

            BackColor = Color.White;
            
            NumberOfMeasures = 0;
            NumberOfTracks = 0;

            hasTracks = false;
            measures = null;
            viewChanged = false;
            spacingChanged = true;

            superParent = f1;
        }

        /// <summary>
        /// Enables the scroll pane and sets up a border layout for the song panel
        /// </summary>
        private void addScrollPanel()
        {
            FlowDirection = FlowDirection.LeftToRight;

            BorderStyle = BorderStyle.Fixed3D;
        }

        /// <summary>
        /// Clears out the panel
        /// </summary>
        private void removeScrollPanel()
        {
            Controls.Clear();
        }

        /// <summary>
        /// Handles what to do when the view of the panel has changed
        /// </summary>
        private void changeView()
        {
            if (!viewChanged) return;

            if (displayOptions.MultiTrackView)
                addScrollPanel();
            else
                removeScrollPanel();
            hasTracks = false;

            spacingChanged = true;
        }

        /// <summary>
        /// Sets the MTP and draws it
        /// </summary>
        public void drawPiece()
        {
            if (hasTracks)
                changeView();
            if (!hasTracks)
            {
                addTrackMeasures();
                hasTracks = true;
            }
            setMeasuresTracksPairs();
            alignBars();
        }

        /// <summary>
        /// Adds a track to the track panel for display
        /// </summary>
        /// <param name="track">The track to add</param>
        private void addTrackPanel(GPTrack track)
        {
            TrackPanel tp = new TrackPanel(displayOptions);
            
            if (measures != null)
            {
                tp.Track = track;
                tp.addBar(measures.Count);
                tp.Measures = measures;
            }
            else
            {
                MessageBox.Show("measures cannot be null (addTrackPanel()) (SongPanel.cs");
                Environment.Exit(0);
            }

            if (displayOptions.MultiTrackView)
            {
                Controls.Add(tp);
                tp.Visible = true;
            }
            else
                tp.AutoScroll = true;
        }

        /// <summary>
        /// Returns the track panel at a given position
        /// </summary>
        /// <param name="index">The index in which to reference with</param>
        /// <returns>The track panel at the index</returns>
        public TrackPanel getTrackPanel(int index)
        {
            TrackPanel tp = null;

            if (index < NumberOfTracks)
            {
                if (displayOptions.MultiTrackView)
                    tp = (TrackPanel)Controls[index];
            }
            else
            {
                MessageBox.Show("problem in getTrackPanel() (SongPanel.cs");
                Environment.Exit(0);
            }
            return tp;
        }

        /// <summary>
        /// Returns the first track panel of the song panel
        /// </summary>
        /// <returns>The first track panel of this song panel</returns>
        private TrackPanel getFirstTrackPanel()
        {
            TrackPanel tp = null;
            count = 0;

            try
            {
                if (displayOptions.MultiTrackView)
                    tp = (TrackPanel)Controls[count];
            }
            catch (IndexOutOfRangeException e)
            {
                MessageBox.Show("index out of range: " + e + "\n\ngetFirstTrackPanel() (SongPanel.cs)");
                Environment.Exit(0);
            }
            return tp;
        }

        /// <summary>
        /// Returns the next track panel of this song panel. The "next" one is indicated by the class
        /// variable "count".
        /// </summary>
        /// <returns></returns>
        private TrackPanel getNextTrackPanel()
        {
            TrackPanel tp = null;
            count++;

            if (count < NumberOfTracks)
            {
                if (displayOptions.MultiTrackView)
                {
                    tp = (TrackPanel)Controls[count];
                }
                else
                {
                    MessageBox.Show("problem in getNextTrackPanel() (SongPanel.cs");
                    Environment.Exit(0);
                }
            }
            return tp;
        }

        /// <summary>
        /// Calculate the minimum duration of the beat of Measure 'm'
        /// </summary>
        /// <param name="m">The measure number of which we are calculating the min duration of</param>
        /// <returns>The duration of the beat in measure m</returns>
        private GPDuration minDurationOfMeasure(int m)
        {
            TrackPanel tp = getFirstTrackPanel();
            BarMtp bmtp;
            GPDuration minDuration = GPDuration.WHOLE;
            GPDuration tempDuration;

            while (tp != null)
            {
                bmtp = tp.getTablatureBar(m);
                tempDuration = bmtp.minDurationOfBeats();

                // The higher the type of duration, the shorter the note is
                if (tempDuration.compareTo(minDuration) < 0)
                    minDuration = tempDuration;
                tp = getNextTrackPanel();
            }
            return minDuration;
        }

        /// <summary>
        /// Propagates the minDuration vertically across the tracks of measure 'm'
        /// </summary>
        /// <param name="minDuration">The minimum duration to set</param>
        /// <param name="m">The measure of which to set the minimum duration</param>
        private void setMinDuration(GPDuration minDuration, int m)
        {
            TrackPanel tp = getFirstTrackPanel();
            BarMtp bmtp;

            while (tp != null)
            {
                bmtp = tp.getTablatureBar(m);
                bmtp.MinDuration = minDuration;
                tp = getNextTrackPanel();
            }
        }

        private void setMinWidth(int min, int m)
        {
            TrackPanel tp = getFirstTrackPanel();
            BarMtp bmtp;

            while (tp != null)
            {
                bmtp = tp.getTablatureBar(m);
                bmtp.MinWidth = min;
                tp = getNextTrackPanel();
            }
        }

        private int getMaxWidth(int m)
        {
            TrackPanel tp = getFirstTrackPanel();
            Bar bar;
            int max = 0;

            while (tp != null)
            {
                bar = tp.getTablatureBar(m);
                max = Math.Max(max, bar.getWidth());
                tp = getNextTrackPanel();
            }

            return max;
        }

        /// <summary>
        /// Aligns the bars. displayNotes must have been called first
        /// </summary>
        private void alignBars()
        {
            GPDuration minDuration;

            if (!spacingChanged) return;

            GPDuration minPieceDuration = GPDuration.WHOLE;

            // Proportional spacing
            for (int i = 0; i < NumberOfMeasures; i++)
            {
                minDuration = minDurationOfMeasure(i);

                if (!displayOptions.MinPieceSpacing) continue;

                if (minDuration.compareTo(minPieceDuration) < 0)
                    minPieceDuration = minDuration;
                else
                    setMinDuration(minDuration, i);
            }

            if (displayOptions.MinPieceSpacing)
                for (int i = 0; i < NumberOfMeasures; i++)
                    setMinDuration(minPieceDuration, i);

            for (int i = 0; i < NumberOfMeasures; i++)
                setMinWidth(getMaxWidth(i), i);

            spacingChanged = false;
        }

        private void setMeasuresTracksPairs()
        {
            List<GPMeasureTrackPair> mtps = Piece.getMeasuresTracksPairs();
            TrackPanel tp;

            if (mtps == null) return;

            int index = 0;

            for (int i = 0; i < NumberOfMeasures; i++)
            {
                tp = getFirstTrackPanel();

                while (tp != null)
                {
                    tp.setMeasureTrackPair(i, mtps[index]);
                    tp = getNextTrackPanel();
                    index++;
                }
            }
        }

        /// <summary>
        /// Add different tracks and measures to a song
        /// </summary>
        private void addTrackMeasures()
        {
            if (measures == null)
                measures = Piece.getMeasures();

            // Now that measures isn't null
            if (measures != null)
            {
                NumberOfMeasures = measures.Count;

                if (NumberOfMeasures > 0)
                {
                    tracks = Piece.getTracks();

                    if (tracks != null)
                    {
                        NumberOfTracks = tracks.Count;

                        if (NumberOfTracks > 0)
                            for (int i = 0; i < NumberOfTracks; i++)
                                addTrackPanel(tracks[i]);
                    }
                }
            }
            
            Width = 205 * NumberOfMeasures;
            Height = 110 * NumberOfTracks;
            superParent.Width = 205 * NumberOfMeasures;
            superParent.Height = 110 * NumberOfTracks;
        }

        public bool readPieceFromFile(GP4InputStream inStr)
        {
            if (inStr != null)
            {
                try
                {
                    Piece = inStr.readPiece();
                    return true;
                }
                catch (IOException e)
                {
                    MessageBox.Show(e.StackTrace + "\n\nreadPieceFromFile() (SongPanel.cs");
                    Environment.Exit(0);
                }
            }
            return false;
        }

        #region Getters and setters
        public bool MultiTrackView
        {
            get { return displayOptions.MultiTrackView; }
        }

        public DisplayOptions DisplayOptions
        {
            set
            {
                displayOptions = value;

                // The top offset is increased by LINES_ON_TOP lines
                displayOptions.TopOffset += LINES_ON_TOP * displayOptions.LineSpacing;

                // The bottom offset is increased by LINES_ON_BOTTOM lines
                displayOptions.BottomOffset += LINES_ON_BOTTOM * displayOptions.BottomOffset;

                //if (displayOptions.MultiTrackView)
                    addScrollPanel();
                //else
                //TODO: Maybe add tab pane option later?
            }
        }

        public GPSong Piece { get; private set; }

        public int NumberOfTracks { get; private set; }

        public int NumberOfMeasures { get; private set; }

        #endregion
    }
}
