﻿using System.Collections.Generic;
using System.Linq;

namespace GuitarTrainer.GuitarProComponents
{
    /// <summary>
    /// This class describes a Beat read from a Guitar Pro file.
    /// A beat can contain several notes and can hold information 
    /// about a chord diagram, a mix table change event, a text marker, 
    /// and effects
    /// </summary>
    class GPBeat
    {
        // The strings played in this beat. 7 strings maximum can be played.
        // The higher the index, the higher the string.
        private readonly bool[] strings;

        // Effects possibly assocated with this beat
        public GPEffectsOnBeat effects;

        // Mix table change possibly associated with this beat
        public GPMixTableChange mixTableChange;

        // Text possibly assocated with this beat
        public string text;

        /// <summary>
        /// Default constructor
        /// </summary>
        public GPBeat()
        {
            ChordDiagram = null;
            DottedNotes = false;
            Duration = GPDuration.QUARTER;
            effects = null;
            IsEmpty = false;
            mixTableChange = null;
            Notes = new List<GPNote>();
            NTuplet = 0;
            IsRestBeat = false;
            strings = new bool[7];
            text = "";
        }

        /// <summary>
        /// Returns a string representation of the Beat
        /// </summary>
        /// <returns>The string representation of this class</returns>
        public override string ToString()
        {
            string toReturn = "\tBeat - BEGIN";

            if (ChordDiagram != null)
                toReturn = "\n\t\t" + ChordDiagram.toString();
            toReturn += "\n\t\tDotted notes: " + DottedNotes;
            toReturn += "\n\t\tDuration: " + Duration;
            if (effects != null)
                toReturn += "\n\t\t" + effects.toString();
            toReturn += "\n\t\tEmpty Beat: " + IsEmpty;
            if (mixTableChange != null)
                toReturn += "\n\t\tMixTableChange" + mixTableChange.toString();
            if (Notes != null)
            {
                toReturn += "\n\t\tNotes: ";
                for (int i = 0; i < Notes.Count; i++)
                    toReturn += "\n\t\t\t" + Notes.ElementAt(i).ToString();
            }
            toReturn += "\n\t\tnTuplet: " + NTuplet;
            toReturn += "\n\t\tRest Beat: " + IsRestBeat;
            toReturn += "\n\t\tStrings played: ";
            for (int i = 6; i >= 0; i--)
            {
                if (!strings[i]) continue;

                toReturn += i;
                if (i - 1 >= 0)
                    toReturn += ",";
            }
            toReturn += "\n\t\tText: " + text;
            toReturn += "\n\tBeat - END";

            return toReturn;
        }

        #region Getters and setters

        public GPChordDiagram ChordDiagram { private get; set; }

        public bool DottedNotes { get; set; }

        public GPDuration Duration { get; set; }

        public bool IsEmpty { private get; set; }

        public List<GPNote> Notes { get; private set; }

        public int NTuplet { get; set; }

        public bool IsRestBeat { get; set; }

        public bool IsNoteBeat
        {
            get { return (!IsEmpty && !IsRestBeat); }
        }
        public bool isStringPlayed(int i)
        {
            return strings[i];
        }

        public void setStrings(int i, bool b)
        {
            strings[i] = b;
        }

        #endregion
    }
}
